<?php

namespace Elementor;

use \Elementor\Controls_Manager;
use \Elementor\Group_Control_Background;
use Elementor\Repeater;
use \Elementor\Core\Schemes\Typography;
use \Elementor\Group_Control_Border;
use \Elementor\Group_Control_Box_Shadow;
use \Elementor\Group_Control_Typography;
use \Elementor\Utils;
use \Elementor\Widget_Base;

if (!defined('ABSPATH')) exit; // If this file is called directly, abort.


class PR_TestimonialSlider_Widget extends Widget_Base
{

    public function get_name() {
		return 'pr-testimonialslider';
	}

    public function get_title()
    {
        return esc_html__('Testimonial Slider', 'pixerex');
    }

    public function get_icon()
    {
        return 'eicon-testimonial-carousel';
    }

    public function get_categories() {
        return [ 'pr-elements' ];
    }

    public function get_keywords()
    {
        return [
            'testimonial slider',
            'testimonial carousel',
            'review',
            'feedback',
            'testimonial carousel',
        ];
    }

    public function get_style_depends()
    {
        return [
            'font-awesome-5-all',
            'font-awesome-4-shim',
        ];
    }

    public function get_script_depends()
    {
        return [
            'font-awesome-4-shim','jquery-swiper','pr-js'
        ];
    }


    protected function register_controls()
    {


        $this->start_controls_section(
            'pre_section_testimonial_content',
            [
                'label' => esc_html__( 'Testimonial Content', 'pixerex' )
            ]
        );

        $repeater = new Repeater();

        $repeater->add_control(
            'pre_testimonial_enable_avatar',
            [
                'label' => esc_html__( 'Display Avatar?', 'pixerex' ),
                'type' => Controls_Manager::SWITCHER,
                'default' => 'yes',
            ]
        );

        $repeater->add_control(
            'pre_testimonial_image',
            [
                'label' => esc_html__( 'Testimonial Avatar', 'pixerex' ),
                'type' => Controls_Manager::MEDIA,
                'default' => [
                    'url' => Utils::get_placeholder_image_src(),
                ],
                'condition' => [
                    'pre_testimonial_enable_avatar' => 'yes',
                ],
            ]
        );

        $repeater->add_control(
            'pre_testimonial_name',
            [
                'label' => esc_html__( 'User Name', 'pixerex' ),
                'type' => Controls_Manager::TEXT,
                'default' => esc_html__( 'John Doe', 'pixerex' ),
                'dynamic' => [ 'active' => true ]
            ]
        );

        $repeater->add_control(
            'pre_testimonial_company_title',
            [
                'name' => 'pre_testimonial_company_title',
                'label' => esc_html__( 'Company Name', 'pixerex' ),
                'type' => Controls_Manager::TEXT,
                'default' => esc_html__( 'Codetic', 'pixerex' ),
                'dynamic' => [ 'active' => true ]
            ]
        );

        $repeater->add_control(
            'pre_testimonial_description',
            [
                'label' => esc_html__( 'Testimonial Description', 'pixerex' ),
                'type' => Controls_Manager::WYSIWYG,
                'default' => esc_html__( 'Add testimonial description here. Edit and place your own text.', 'pixerex' ),
            ]
        );

        $repeater->add_control(
            'pre_testimonial_enable_rating',
            [
                'label' => esc_html__( 'Display Rating?', 'pixerex' ),
                'type' => Controls_Manager::SWITCHER,
                'default' => 'yes',
            ]
        );

        $repeater->add_control(
            'pre_testimonial_rating_number',
            [
                'label'       => __( 'Rating Number', 'pixerex' ),
                'type' => Controls_Manager::SELECT,
                'default' => 'rating-five',
                'options' => [
                    'rating-one'  => __( '1', 'pixerex' ),
                    'rating-two' => __( '2', 'pixerex' ),
                    'rating-three' => __( '3', 'pixerex' ),
                    'rating-four' => __( '4', 'pixerex' ),
                    'rating-five'   => __( '5', 'pixerex' ),
                ],
                'condition' => [
                    'pre_testimonial_enable_rating' => 'yes',
                ],
            ]
        );

		$this->add_control(
			'pre_testimonial_slider_item',
			[
				'type' => Controls_Manager::REPEATER,
				'default' => [
					[
						'pre_testimonial_name' => 'John Doe',
					],
					[
						'pre_testimonial_name' => 'Jane Doe',
					],

				],
				'fields' => $repeater->get_controls(),
				'title_field' => 'Testimonial Item',
			]
        );
        
		$this->add_control(
			'pre_testimonial_item_description_height',
			[
				'label' => esc_html__( 'Description Height', 'pixerex'),
				'type' => Controls_Manager::SLIDER,
				'size_units'	=> ['px', '%', 'em'],
				'range' => [
					'px' => [ 'max' => 500 ],
					'%'	=> [ 'max'	=> 100 ]
				],
			]
		);
		$this->end_controls_section();

		/**
         * Content Tab: Carousel Settings
         */
        $this->start_controls_section(
            'section_additional_options',
            [
                'label'                 => __('Carousel Settings', 'pixerex'),
            ]
        );

        $this->add_control(
            'carousel_effect',
            [
                'label'                 => __('Effect', 'pixerex'),
                'description'           => __('Sets transition effect', 'pixerex'),
                'type'                  => Controls_Manager::SELECT,
                'default'               => 'slide',
                'options'               => [
                    'slide'     => __('Slide', 'pixerex'),
                    'fade'      => __('Fade', 'pixerex'),
                    'cube'      => __('Cube', 'pixerex'),
                    'coverflow' => __('Coverflow', 'pixerex'),
                    'flip'      => __('Flip', 'pixerex'),
                ],
            ]
        );

        $this->add_responsive_control(
            'items',
            [
                'label'                 => __('Visible Items', 'pixerex'),
                'type'                  => Controls_Manager::SLIDER,
                'default'               => ['size' => 1],
                'tablet_default'        => ['size' => 1],
                'mobile_default'        => ['size' => 1],
                'range'                 => [
                    'px' => [
                        'min'   => 1,
                        'max'   => 10,
                        'step'  => 1,
                    ],
                ],
                'size_units'            => '',
                'condition' => [
                    'carousel_effect' => ['slide', 'coverflow']
                ]
            ]
        );

        $this->add_responsive_control(
            'margin',
            [
                'label'                 => __('Items Gap', 'pixerex'),
                'type'                  => Controls_Manager::SLIDER,
                'default'               => ['size' => 10],
                'range'                 => [
                    'px' => [
                        'min'   => 0,
                        'max'   => 100,
                        'step'  => 1,
                    ],
                ],
                'size_units'            => '',
                'condition' => [
                    'carousel_effect' => ['slide', 'coverflow']
                ]
            ]
        );

        $this->add_control(
            'slider_speed',
            [
                'label'                 => __('Slider Speed', 'pixerex'),
                'description'           => __('Duration of transition between slides (in ms)', 'pixerex'),
                'type'                  => Controls_Manager::SLIDER,
                'default'               => ['size' => 1000],
                'range'                 => [
                    'px' => [
                        'min'   => 100,
                        'max'   => 3000,
                        'step'  => 1,
                    ],
                ],
                'size_units'            => '',
            ]
        );

        $this->add_control(
            'autoplay',
            [
                'label'                 => __('Autoplay', 'pixerex'),
                'type'                  => Controls_Manager::SWITCHER,
                'default'               => 'yes',
                'label_on'              => __('Yes', 'pixerex'),
                'label_off'             => __('No', 'pixerex'),
                'return_value'          => 'yes',
            ]
        );

        $this->add_control(
            'autoplay_speed',
            [
                'label'                 => __('Autoplay Speed', 'pixerex'),
                'type'                  => Controls_Manager::SLIDER,
                'default'               => ['size' => 2000],
                'range'                 => [
                    'px' => [
                        'min'   => 500,
                        'max'   => 5000,
                        'step'  => 1,
                    ],
                ],
                'size_units'            => '',
                'condition'         => [
                    'autoplay'      => 'yes',
                ],
            ]
        );

        $this->add_control(
            'pause_on_hover',
            [
                'label'                 => __('Pause On Hover', 'pixerex'),
                'type'                  => Controls_Manager::SWITCHER,
                'default'               => '',
                'label_on'              => __('Yes', 'pixerex'),
                'label_off'             => __('No', 'pixerex'),
                'return_value'          => 'yes',
                'condition'         => [
                    'autoplay'      => 'yes',
                ],
            ]
        );

        $this->add_control(
            'infinite_loop',
            [
                'label'                 => __('Infinite Loop', 'pixerex'),
                'type'                  => Controls_Manager::SWITCHER,
                'default'               => 'yes',
                'label_on'              => __('Yes', 'pixerex'),
                'label_off'             => __('No', 'pixerex'),
                'return_value'          => 'yes',
            ]
        );

        $this->add_control(
            'grab_cursor',
            [
                'label'                 => __('Grab Cursor', 'pixerex'),
                'description'           => __('Shows grab cursor when you hover over the slider', 'pixerex'),
                'type'                  => Controls_Manager::SWITCHER,
                'default'               => '',
                'label_on'          => __('Show', 'pixerex'),
                'label_off'         => __('Hide', 'pixerex'),
                'return_value'      => 'yes',
            ]
        );

        $this->add_control(
            'navigation_heading',
            [
                'label'                 => __('Navigation', 'pixerex'),
                'type'                  => Controls_Manager::HEADING,
                'separator'             => 'before',
            ]
        );

        $this->add_control(
            'arrows',
            [
                'label'                 => __('Arrows', 'pixerex'),
                'type'                  => Controls_Manager::SWITCHER,
                'default'               => 'yes',
                'label_on'              => __('Yes', 'pixerex'),
                'label_off'             => __('No', 'pixerex'),
                'return_value'          => 'yes',
            ]
        );

	    $this->add_control(
		    'navigation_arrow_position',
		    [
			    'label'                 => __('Navigation arrow position', 'pixerex'),
			    'type'                  => Controls_Manager::SELECT2,
			    'default' => 'default',
			    'options' => [
				    'default'            => __( 'Default', 'plugin-domain' ),
				    'outside-of-the-box' => __( 'Outside of the box', 'plugin-domain' ),
			    ],
			    'condition' => [
				    'arrows'    => 'yes'
			    ]
		    ]
	    );

        $this->add_control(
            'dots',
            [
                'label'                 => __('Dots', 'pixerex'),
                'type'                  => Controls_Manager::SWITCHER,
                'default'               => 'yes',
                'label_on'              => __('Yes', 'pixerex'),
                'label_off'             => __('No', 'pixerex'),
                'return_value'          => 'yes',
            ]
        );

        $this->add_control(
            'image_dots',
            [
                'label'                 => __('Image Dots', 'pixerex'),
                'type'                  => Controls_Manager::SWITCHER,
                'label_on'              => __('Yes', 'pixerex'),
                'label_off'             => __('No', 'pixerex'),
                'return_value'          => 'yes',
                'condition' => [
	                'dots'    => 'yes'
                ]
            ]
        );


        $this->add_responsive_control(
            'image_dots_visibility',
            [
                'label' => __('Image Dots Visibility', 'pixerex'),
                'type' => \Elementor\Controls_Manager::SWITCHER,
                'label_on' => __('Show', 'pixerex'),
                'label_off' => __('Hide', 'pixerex'),
                'return_value' => 'yes',
                'default' => 'yes',
                'condition' => [
                    'image_dots'    => 'yes'
                ]
            ]
        );

        $this->add_control('image_dots_position', [
            'label' => esc_html__('Image Dots Position', 'pixerex'),
            'type' => Controls_Manager::CHOOSE,
            'label_block' => false,
            'default' => 'below',
            'options' => [
                    'above' => [
                            'title' => esc_html__('Above', 'pixerex'),
                            'icon' => 'eicon-v-align-top',
                    ],
                    'below' => [
                            'title' => esc_html__('Below', 'pixerex'),
                            'icon' => 'eicon-v-align-bottom',
                    ],
            ],
            'condition' => [
                'image_dots'    => 'yes'
            ]
    ]);


        $this->end_controls_section();


        $this->start_controls_section(
            'pre_section_testimonial_styles_general',
            [
                'label' => esc_html__('Testimonial Styles', 'pixerex'),
                'tab' => Controls_Manager::TAB_STYLE
            ]
        );

        $this->add_control(
            'pre_testimonial_layout',
            [
                'label' => __( 'Layout', 'pixerex'),
                'type' => Controls_Manager::SELECT,
                'default' => 'above',
                'options' => [
                    'above'   => __('Image Above', 'pixerex'),
                    'top'   => __('Image Top', 'pixerex'),
                    'below'   => __('Image Below', 'pixerex'), 
                ],

            ]
        );

        $this->start_controls_tabs('testimonial_bg_tabs');

            // start normal state tab
			$this->start_controls_tab('testimonial_tab_normalbg', [
				'label' => esc_html__( 'Normal', 'pixerex')
			]);

            $this->add_group_control(
                \Elementor\Group_Control_Background::get_type(),
                [
                    'name' => 'pre_testimonial_background',
                    'label' => __('Gradient Background', 'pixerex'),
                    'types' => ['classic', 'gradient'],
                    'selector' => '{{WRAPPER}} .pre-testimonial-item  .pre-testimonial-item-inner',
                ]
            );

            $this->end_controls_tab();

			// start hover state tab
			$this->start_controls_tab('testimonial_tab_hoverbg', [
				'label' => esc_html__( 'Hover', 'pixerex')
			]);

            $this->add_group_control(
                \Elementor\Group_Control_Background::get_type(),
                [
                    'name' => 'pre_testimonial_item_hover_background',
                    'label' => __('Background', 'pixerex'),
                    'types' => ['classic', 'gradient'],
                    'selector' => '{{WRAPPER}} .pre-testimonial-item:hover .pre-testimonial-item-inner',
                ]
            );

            $this->end_controls_tab();

        $this->end_controls_tabs();

        $this->add_control(
            'pre_testimonial_alignment',
            [
                'label' => esc_html__('Set Alignment', 'pixerex'),
                'type' => Controls_Manager::CHOOSE,
                'label_block' => true,
                'default' => 'center',
				'options' => [
					'left'    => [
						'title' => esc_html__( 'Left', 'pixerex' ),
						'icon' => 'eicon-text-align-left',
					],
					'center' => [
						'title' => esc_html__( 'Center', 'pixerex' ),
						'icon' => 'eicon-text-align-center',
					],
					'right' => [
						'title' => esc_html__( 'Right', 'pixerex' ),
						'icon' => 'eicon-text-align-right',
					],
				],
                'selectors' => [
					'{{WRAPPER}} .pre-testimonial-item-inner' => 'text-align: {{VALUE}}',
				],
            ]
        );

        $this->add_responsive_control(
            'pre_testimonial_margin',
            [
                'label' => esc_html__('Margin', 'pixerex'),
                'description' => 'Need to refresh the page to see the change properly',
                'type' => Controls_Manager::DIMENSIONS,
                'size_units' => ['px', '%', 'em'],
                'selectors' => [
                    '{{WRAPPER}} .pre-testimonial-item .pre-testimonial-item-inner' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->add_responsive_control(
            'pre_testimonial_padding',
            [
                'label' => esc_html__('Padding', 'pixerex'),
                'type' => Controls_Manager::DIMENSIONS,
                'size_units' => ['px', '%', 'em'],
                'selectors' => [
                    '{{WRAPPER}} .pre-testimonial-item .pre-testimonial-item-inner' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->add_group_control(
            Group_Control_Border::get_type(),
            [
                'name' => 'pre_testimonial_border',
                'label' => esc_html__('Border', 'pixerex'),
                'selector' => '{{WRAPPER}} .pre-testimonial-item .pre-testimonial-item-inner',
            ]
        );

        $this->add_control(
            'pre_testimonial_border_radius',
            [
                'label' => esc_html__('Border Radius', 'pixerex'),
                'type' => Controls_Manager::DIMENSIONS,
                'selectors' => [
                    '{{WRAPPER}} .pre-testimonial-item  .pre-testimonial-item-inner' => 'border-radius: {{TOP}}px {{RIGHT}}px {{BOTTOM}}px {{LEFT}}px;',
                ],
            ]
        );
        $this->add_group_control(
            \Elementor\Group_Control_Box_Shadow::get_type(),
            [
                'name' => 'pre_testimonial_item_shadow',
                'label' => __('Shadow', 'pixerex'),
                'selector' => '{{WRAPPER}} .pre-testimonial-item  .pre-testimonial-item-inner',
            ]
        );

        $this->end_controls_section();

        /**
         * Image Style Section
         */
        $this->start_controls_section(
            'pre_section_testimonial_image_styles',
            [
                'label' => esc_html__('Image', 'pixerex'),
                'tab' => Controls_Manager::TAB_STYLE
            ]
        );

        $this->add_control(
            'pre_testimonial_img_position',
            [
                'label' => __( 'Image Position', 'pixerex' ),
                'type' => Controls_Manager::SELECT,
                'options' => [
                    'pre-testimonial-img-top' => esc_html__( 'Top', 'pixerex' ),
                    'pre-testimonial-img-aside' => esc_html__( 'Aside', 'pixerex' ),
                ],
                'default' => 'pr-testimonial-img-top',

                'condition' => [
                    'pre_testimonial_layout' => ['above', 'below']
                ]

            ]
        );

        $this->add_responsive_control(
            'pre_testimonial_image_width',
            [
                'label' => esc_html__('Image Width', 'pixerex'),
                'type' => Controls_Manager::SLIDER,
                'default' => [
                    'size' => 150,
                    'unit' => 'px',
                ],
                'range' => [
                    '%' => [
                        'min' => 0,
                        'max' => 100,
                    ],
                    'px' => [
                        'min' => 0,
                        'max' => 1000,
                    ],
                ],
                'size_units' => ['%', 'px'],
                'selectors' => [
                    '{{WRAPPER}} .pre-testimonial-image img' => 'width:{{SIZE}}{{UNIT}};',
                ],
            ]
        );


        $this->add_responsive_control(
            'pre_testimonial_image_margin',
            [
                'label' => esc_html__('Margin', 'pixerex'),
                'type' => Controls_Manager::DIMENSIONS,
                'size_units' => ['px', '%'],
                'selectors' => [
                    '{{WRAPPER}} .pre-testimonial-image img' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->add_responsive_control(
            'pre_testimonial_image_padding',
            [
                'label' => esc_html__('Padding', 'pixerex'),
                'type' => Controls_Manager::DIMENSIONS,
                'size_units' => ['px', '%', 'em'],
                'selectors' => [
                    '{{WRAPPER}} .pre-testimonial-image img' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );


        $this->add_group_control(
            Group_Control_Border::get_type(),
            [
                'name' => 'pre_testimonial_image_border',
                'label' => esc_html__('Border', 'pixerex'),
                'selector' => '{{WRAPPER}} .pre-testimonial-image img',
            ]
        );

        $this->add_control(
            'pre_testimonial_image_rounded',
            [
                'label' => esc_html__('Rounded Avatar?', 'pixerex'),
                'type' => Controls_Manager::SWITCHER,
                'return_value' => 'testimonial-avatar-rounded',
                'default' => '',
            ]
        );


        $this->add_control(
            'pre_testimonial_image_border_radius',
            [
                'label' => esc_html__('Border Radius', 'pixerex'),
                'type' => Controls_Manager::DIMENSIONS,
                'selectors' => [
                    '{{WRAPPER}} .pre-testimonial-image img' => 'border-radius: {{TOP}}px {{RIGHT}}px {{BOTTOM}}px {{LEFT}}px;',
                ],
                'condition' => [
                    'pre_testimonial_image_rounded!' => 'testimonial-avatar-rounded',
                ],
            ]
        );

        $this->end_controls_section();

        /**
         * Style Tab: Testimonial content
         */

        $this->start_controls_section(
            'pre_testimonial_dsc_style',
            [
                'label' => esc_html__('Description', 'pixerex'),
                'tab' => Controls_Manager::TAB_STYLE
            ]
        );

        $this->add_control(
            'pre_testimonial_description_color',
            [
                'label' => esc_html__('Description Color', 'pixerex'),
                'type' => Controls_Manager::COLOR,
                'default' => '#7a7a7a',
                'selectors' => [
                    '{{WRAPPER}} .pre-testimonial-content .pre-testimonial-text' => 'color: {{VALUE}};',
                ],
            ]
        );

        $this->add_control(
            'pre_testimonial_hover_desc_color',
            [
                'label' => esc_html__('Description Hover Color', 'pixerex'),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .pre-testimonial-item:hover .pre-testimonial-content .pre-testimonial-text' => 'color: {{VALUE}};',
                ],
            ]
        );

        $this->add_group_control(
            Group_Control_Typography::get_type(),
            [
                'name' => 'pre_testimonial_description_typography',
                'selector' => '{{WRAPPER}} .pre-testimonial-content .pre-testimonial-text',
            ]
        );

        $this->add_responsive_control(
            'pre_testimonial_dsc_margin',
            [
                'label' => esc_html__('Margin', 'pixerex'),
                'type' => Controls_Manager::DIMENSIONS,
                'size_units' => ['px', '%'],
                'selectors' => [
                    '{{WRAPPER}} .pre-testimonial-content .pre-testimonial-text' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->end_controls_section();

        $this->start_controls_section(
            'pre_testimonial_name_style',
            [
                'label' => esc_html__('Name', 'pixerex'),
                'tab' => Controls_Manager::TAB_STYLE
            ]
        );

        $this->add_control(
            'pre_testimonial_name_color',
            [
                'label' => esc_html__('Name Color', 'pixerex'),
                'type' => Controls_Manager::COLOR,
                'default' => '#272727',
                'selectors' => [
                    '{{WRAPPER}} .pre-testimonial-content .pre-testimonial-user' => 'color: {{VALUE}};',
                ],
            ]
        );

        $this->add_control(
            'pre_testimonial_hover_name_color',
            [
                'label' => esc_html__('Name Hover Color', 'pixerex'),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .pre-testimonial-item:hover .pre-testimonial-content .pre-testimonial-user' => 'color: {{VALUE}};',
                ],
            ]
        );

        $this->add_group_control(
            Group_Control_Typography::get_type(),
            [
                'name' => 'pre_testimonial_name_typography',
                'selector' => '{{WRAPPER}} .pre-testimonial-content .pre-testimonial-user',
            ]
        );
        
        $this->add_responsive_control(
            'pre_testimonial_name_margin',
            [
                'label' => esc_html__('Margin', 'pixerex'),
                'type' => Controls_Manager::DIMENSIONS,
                'size_units' => ['px', '%'],
                'selectors' => [
                    '{{WRAPPER}} .pre-testimonial-content .pre-testimonial-user' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->end_controls_section();

        $this->start_controls_section(
            'pre_testimonial_company_style',
            [
                'label' => esc_html__('Company', 'pixerex'),
                'tab' => Controls_Manager::TAB_STYLE
            ]
        );

        $this->add_control(
            'pre_testimonial_company_color',
            [
                'label' => esc_html__('Company Color', 'pixerex'),
                'type' => Controls_Manager::COLOR,
                'default' => '#272727',
                'selectors' => [
                    '{{WRAPPER}} .pre-testimonial-content .pre-testimonial-user-company' => 'color: {{VALUE}};',
                ],
            ]
        );

        $this->add_control(
            'pre_testimonial_hover_company_color',
            [
                'label' => esc_html__('Company Color', 'pixerex'),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .pre-testimonial-item:hover .pre-testimonial-content .pre-testimonial-user-company' => 'color: {{VALUE}};',
                ],
            ]
        );

        $this->add_group_control(
            Group_Control_Typography::get_type(),
            [
                'name' => 'pre_testimonial_position_typography',
                'selector' => '{{WRAPPER}} .pre-testimonial-content .pre-testimonial-user-company',
            ]
        );
        
        $this->add_responsive_control(
            'pre_testimonial_company_margin',
            [
                'label' => esc_html__('Margin', 'pixerex'),
                'type' => Controls_Manager::DIMENSIONS,
                'size_units' => ['px', '%'],
                'selectors' => [
                    '{{WRAPPER}} .pre-testimonial-content .pre-testimonial-user-company' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->end_controls_section();

        /**
         * Style Tab: Rating
         */

        $this->start_controls_section(
			'section_rating_style',
			[
				'label' => __( 'Rating', 'pixerex' ),
				'tab' => Controls_Manager::TAB_STYLE,
			]
		);

		$this->add_control(
			'pre_testimonial_star_size',
			[
				'label' => __( 'Size', 'pixerex' ),
				'type' => Controls_Manager::SLIDER,
				'range' => [
					'px' => [
						'min' => 0,
						'max' => 100,
					],
				],
				'selectors' => [
					'{{WRAPPER}} .testimonial-star-rating li i' => 'font-size: {{SIZE}}{{UNIT}}',
				],
				'separator' => 'before',
			]
		);

		$this->add_control(
			'pre_testimonial_star_space',
			[
				'label' => __( 'Spacing', 'pixerex' ),
				'type' => Controls_Manager::SLIDER,
				'range' => [
					'px' => [
						'min' => 0,
						'max' => 50,
					],
				],
				'selectors' => [
					'body:not(.rtl) {{WRAPPER}} .testimonial-star-rating li:not(:last-of-type)' => 'margin-right: {{SIZE}}{{UNIT}}',
					'body.rtl {{WRAPPER}} .testimonial-star-rating li:not(:last-of-type)' => 'margin-left: {{SIZE}}{{UNIT}}',
				],
			]
		);

        $this->add_responsive_control(
            'pre_testimonial_rating_margin',
            [
                'label' => esc_html__('Margin', 'pixerex'),
                'type' => Controls_Manager::DIMENSIONS,
                'size_units' => ['px', '%'],
                'selectors' => [
                    '{{WRAPPER}} .pre-testimonial-content .testimonial-star-rating' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

		$this->end_controls_section();

        /**
         * Style Tab: Arrows
         */
        $this->start_controls_section(
            'section_arrows_style',
            [
                'label'                 => __('Arrows', 'pixerex'),
                'tab'                   => Controls_Manager::TAB_STYLE,
                'condition'             => [
                    'arrows'        => 'yes',
                ],
            ]
        );

        $this->add_control(
            'arrow',
            [
                'label'                 => __('Choose Arrow', 'pixerex'),
                'type'                  => Controls_Manager::SELECT,
                'label_block'           => true,
                'default'               => 'fa fa-angle-right',
                'options'               => [
                    'fa fa-angle-right'             => __('Angle', 'pixerex'),
                    'fa fa-angle-double-right'      => __('Double Angle', 'pixerex'),
                    'fa fa-chevron-right'           => __('Chevron', 'pixerex'),
                    'fa fa-chevron-circle-right'    => __('Chevron Circle', 'pixerex'),
                    'fa fa-arrow-right'             => __('Arrow', 'pixerex'),
                    'fa fa-long-arrow-right'        => __('Long Arrow', 'pixerex'),
                    'fa fa-caret-right'             => __('Caret', 'pixerex'),
                    'fa fa-caret-square-o-right'    => __('Caret Square', 'pixerex'),
                    'fa fa-arrow-circle-right'      => __('Arrow Circle', 'pixerex'),
                    'fa fa-arrow-circle-o-right'    => __('Arrow Circle O', 'pixerex'),
                    'fa fa-toggle-right'            => __('Toggle', 'pixerex'),
                    'fa fa-hand-o-right'            => __('Hand', 'pixerex'),
                ],
            ]
        );

        $this->add_responsive_control(
            'arrows_width',
            [
                'label'                 => __('Arrows Width', 'pixerex'),
                'type'                  => Controls_Manager::SLIDER,
                'default'               => ['size' => '27'],
                'range'                 => [
                    'px' => [
                        'min'   => 15,
                        'max'   => 200,
                        'step'  => 1,
                    ],
                ],
                'size_units'            => ['px'],
                'selectors'             => [
                    '{{WRAPPER}} .swiper-container-wrap .swiper-button-next, {{WRAPPER}} .swiper-container-wrap .swiper-button-prev' => 'width: {{SIZE}}{{UNIT}};',
                ],
            ]
        );
        $this->add_responsive_control(
            'arrows_height',
            [
                'label'                 => __('Arrows Height', 'pixerex'),
                'type'                  => Controls_Manager::SLIDER,
                'default'               => ['size' => '44'],
                'range'                 => [
                    'px' => [
                        'min'   => 15,
                        'max'   => 200,
                        'step'  => 1,
                    ],
                ],
                'size_units'            => ['px'],
                'selectors'             => [
                    '{{WRAPPER}} .swiper-container-wrap .swiper-button-next, {{WRAPPER}} .swiper-container-wrap .swiper-button-prev' => 'height: {{SIZE}}{{UNIT}};',
                ],
            ]
        );

        $this->add_responsive_control(
            'arrows_size',
            [
                'label'                 => __('Arrows Font Size', 'pixerex'),
                'type'                  => Controls_Manager::SLIDER,
                'default'               => ['size' => '22'],
                'range'                 => [
                    'px' => [
                        'min'   => 15,
                        'max'   => 100,
                        'step'  => 1,
                    ],
                ],
                'size_units'            => ['px'],
                'selectors'             => [
                    '{{WRAPPER}} .swiper-container-wrap .swiper-button-next, {{WRAPPER}} .swiper-container-wrap .swiper-button-prev' => 'font-size: {{SIZE}}{{UNIT}};',
                ],
            ]
        );

        $this->add_responsive_control(
            'left_arrow_position',
            [
                'label'                 => __('Align Left Arrow', 'pixerex'),
                'type'                  => Controls_Manager::SLIDER,
                'range'                 => [
                    'px' => [
                        'min'   => -100,
                        'max'   => 40,
                        'step'  => 1,
                    ],
                ],
                'size_units'            => ['px'],
                'selectors'         => [
                    '{{WRAPPER}} .swiper-container-wrap .swiper-button-prev' => 'left: {{SIZE}}{{UNIT}};',
                ],
            ]
        );

        $this->add_responsive_control(
            'right_arrow_position',
            [
                'label'                 => __('Align Right Arrow', 'pixerex'),
                'type'                  => Controls_Manager::SLIDER,
                'range'                 => [
                    'px' => [
                        'min'   => -100,
                        'max'   => 40,
                        'step'  => 1,
                    ],
                ],
                'size_units'            => ['px'],
                'selectors'         => [
                    '{{WRAPPER}} .swiper-container-wrap .swiper-button-next' => 'right: {{SIZE}}{{UNIT}};',
                ],
            ]
        );

        $this->start_controls_tabs('tabs_arrows_style');

        $this->start_controls_tab(
            'tab_arrows_normal',
            [
                'label'                 => __('Normal', 'pixerex'),
            ]
        );

        $this->add_control(
            'arrows_bg_color_normal',
            [
                'label'                 => __('Background Color', 'pixerex'),
                'type'                  => Controls_Manager::COLOR,
                'default'               => '',
                'selectors'             => [
                    '{{WRAPPER}} .swiper-container-wrap .swiper-button-next, {{WRAPPER}} .swiper-container-wrap .swiper-button-prev' => 'background-color: {{VALUE}};',
                ],
            ]
        );

        $this->add_control(
            'arrows_color_normal',
            [
                'label'                 => __('Color', 'pixerex'),
                'type'                  => Controls_Manager::COLOR,
                'default'               => '',
                'selectors'             => [
                    '{{WRAPPER}} .swiper-container-wrap .swiper-button-next, {{WRAPPER}} .swiper-container-wrap .swiper-button-prev' => 'color: {{VALUE}};',
                ],
            ]
        );

        $this->add_group_control(
            Group_Control_Border::get_type(),
            [
                'name'                  => 'arrows_border_normal',
                'label'                 => __('Border', 'pixerex'),
                'placeholder'           => '1px',
                'default'               => '1px',
                'selector'              => '{{WRAPPER}} .swiper-container-wrap .swiper-button-next, {{WRAPPER}} .swiper-container-wrap .swiper-button-prev'
            ]
        );

        $this->add_control(
            'arrows_border_radius_normal',
            [
                'label'                 => __('Border Radius', 'pixerex'),
                'type'                  => Controls_Manager::DIMENSIONS,
                'size_units'            => ['px', '%'],
                'selectors'             => [
                    '{{WRAPPER}} .swiper-container-wrap .swiper-button-next, {{WRAPPER}} .swiper-container-wrap .swiper-button-prev' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->end_controls_tab();

        $this->start_controls_tab(
            'tab_arrows_hover',
            [
                'label'                 => __('Hover', 'pixerex'),
            ]
        );

        $this->add_control(
            'arrows_bg_color_hover',
            [
                'label'                 => __('Background Color', 'pixerex'),
                'type'                  => Controls_Manager::COLOR,
                'default'               => '',
                'selectors'             => [
                    '{{WRAPPER}} .swiper-container-wrap .swiper-button-next:hover, {{WRAPPER}} .swiper-container-wrap .swiper-button-prev:hover' => 'background-color: {{VALUE}};',
                ],
            ]
        );

        $this->add_control(
            'arrows_color_hover',
            [
                'label'                 => __('Color', 'pixerex'),
                'type'                  => Controls_Manager::COLOR,
                'default'               => '',
                'selectors'             => [
                    '{{WRAPPER}} .swiper-container-wrap .swiper-button-next:hover, {{WRAPPER}} .swiper-container-wrap .swiper-button-prev:hover' => 'color: {{VALUE}};',
                ],
            ]
        );

        $this->add_control(
            'arrows_border_color_hover',
            [
                'label'                 => __('Border Color', 'pixerex'),
                'type'                  => Controls_Manager::COLOR,
                'default'               => '',
                'selectors'             => [
                    '{{WRAPPER}} .swiper-container-wrap .swiper-button-next:hover, {{WRAPPER}} .swiper-container-wrap .swiper-button-prev:hover' => 'border-color: {{VALUE}};',
                ],
            ]
        );

        $this->end_controls_tab();

        $this->end_controls_tabs();

        $this->add_responsive_control(
            'arrows_padding',
            [
                'label'                 => __('Padding', 'pixerex'),
                'type'                  => Controls_Manager::DIMENSIONS,
                'size_units'            => ['px', '%'],
                'selectors'             => [
                    '{{WRAPPER}} .swiper-container-wrap .swiper-button-next, {{WRAPPER}} .swiper-container-wrap .swiper-button-prev' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
                'separator'             => 'before',
            ]
        );

        $this->end_controls_section();

        /**
         * Style Tab: Dots
         */
        $this->start_controls_section(
            'section_dots_style',
            [
                'label'                 => __('Dots', 'pixerex'),
                'tab'                   => Controls_Manager::TAB_STYLE,
                'condition'             => [
                    'dots'      => 'yes',
                ],
            ]
        );

        $this->add_control(
            'dots_position',
            [
                'label'                 => __('Position', 'pixerex'),
                'type'                  => Controls_Manager::SELECT,
                'options'               => [
                    'inside'     => __('Inside', 'pixerex'),
                    'outside'    => __('Outside', 'pixerex'),
                ],
                'default'               => 'outside',
            ]
        );

        $this->add_responsive_control(
            'dots_size',
            [
                'label'                 => __('Size', 'pixerex'),
                'type'                  => Controls_Manager::SLIDER,
                'range'                 => [
                    'px' => [
                        'min'   => 2,
                        'max'   => 40,
                        'step'  => 1,
                    ],
                ],
                'size_units'            => '',
                'selectors'             => [
                    '{{WRAPPER}} .swiper-container-wrap .swiper-pagination-bullet' => 'height: {{SIZE}}{{UNIT}}; width: {{SIZE}}{{UNIT}}',
                ],
            ]
        );

        $this->add_responsive_control(
            'dots_spacing',
            [
                'label'                 => __('Spacing', 'pixerex'),
                'type'                  => Controls_Manager::SLIDER,
                'range'                 => [
                    'px' => [
                        'min'   => 1,
                        'max'   => 30,
                        'step'  => 1,
                    ],
                ],
                'size_units'            => '',
                'selectors'             => [
                    '{{WRAPPER}} .swiper-container-wrap .swiper-pagination-bullet' => 'margin-left: {{SIZE}}{{UNIT}}; margin-right: {{SIZE}}{{UNIT}}',
                ],
            ]
        );

        $this->start_controls_tabs('tabs_dots_style');

        $this->start_controls_tab(
            'tab_dots_normal',
            [
                'label'                 => __('Normal', 'pixerex'),
            ]
        );

        $this->add_control(
            'dots_color_normal',
            [
                'label'                 => __('Color', 'pixerex'),
                'type'                  => Controls_Manager::COLOR,
                'default'               => '',
                'selectors'             => [
                    '{{WRAPPER}} .swiper-container-wrap .swiper-pagination-bullet' => 'background: {{VALUE}};',
                ],
            ]
        );

        $this->add_control(
            'active_dot_color_normal',
            [
                'label'                 => __('Active Color', 'pixerex'),
                'type'                  => Controls_Manager::COLOR,
                'default'               => '',
                'selectors'             => [
                    '{{WRAPPER}} .swiper-container-wrap .swiper-pagination-bullet-active' => 'background: {{VALUE}};',
                ],
            ]
        );

        $this->add_group_control(
            Group_Control_Border::get_type(),
            [
                'name'                  => 'dots_border_normal',
                'label'                 => __('Border', 'pixerex'),
                'placeholder'           => '1px',
                'default'               => '1px',
                'selector'              => '{{WRAPPER}} .swiper-container-wrap .swiper-pagination-bullet',
            ]
        );

        $this->add_control(
            'dots_border_radius_normal',
            [
                'label'                 => __('Border Radius', 'pixerex'),
                'type'                  => Controls_Manager::DIMENSIONS,
                'size_units'            => ['px', '%'],
                'selectors'             => [
                    '{{WRAPPER}} .swiper-container-wrap .swiper-pagination-bullet' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->add_responsive_control(
            'dots_padding',
            [
                'label'                 => __('Padding', 'pixerex'),
                'type'                  => Controls_Manager::DIMENSIONS,
                'size_units'            => ['px', 'em', '%'],
                'allowed_dimensions'    => 'vertical',
                'placeholder'           => [
                    'top'      => '',
                    'right'    => 'auto',
                    'bottom'   => '',
                    'left'     => 'auto',
                ],
                'selectors'             => [
                    '{{WRAPPER}} .swiper-container-wrap .swiper-pagination-bullets' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->end_controls_tab();

        $this->start_controls_tab(
            'tab_dots_hover',
            [
                'label'                 => __('Hover', 'pixerex'),
            ]
        );

        $this->add_control(
            'dots_color_hover',
            [
                'label'                 => __('Color', 'pixerex'),
                'type'                  => Controls_Manager::COLOR,
                'default'               => '',
                'selectors'             => [
                    '{{WRAPPER}} .swiper-container-wrap .swiper-pagination-bullet:hover' => 'background: {{VALUE}};',
                ],
            ]
        );

        $this->add_control(
            'dots_border_color_hover',
            [
                'label'                 => __('Border Color', 'pixerex'),
                'type'                  => Controls_Manager::COLOR,
                'default'               => '',
                'selectors'             => [
                    '{{WRAPPER}} .swiper-container-wrap .swiper-pagination-bullet:hover' => 'border-color: {{VALUE}};',
                ],
            ]
        );

        $this->end_controls_tab();

        $this->end_controls_tabs();

        $this->end_controls_section();

        /**
         * Image Dots
         */
        $this->start_controls_section(
            'section_image_dots_style',
            [
                'label'                 => __('Images Dots', 'pixerex'),
                'tab'                   => Controls_Manager::TAB_STYLE,
                'condition'             => [
                    'image_dots'      => 'yes',
                ],
            ]
        );
        $this->add_responsive_control(
            'pre_image_dots_width',
            [
                'label' => __('Width', 'pixerex'),
                'type' => Controls_Manager::SLIDER,
                'size_units' => ['px', '%'],
                'range' => [
                    'px' => [
                        'min' => 0,
                        'max' => 1000,
                        'step' => 5,
                    ],
                    '%' => [
                        'min' => 0,
                        'max' => 100,
                    ],
                ],
                'devices' => ['desktop', 'tablet', 'mobile'],
                'default' => [
                    'unit' => 'px',
                    'size' => 350,
                ],
                'selectors' => [
                    '{{WRAPPER}} .pre-testimonial-gallary-pagination' => 'width: {{SIZE}}{{UNIT}} !important;',
                ],
            ]
        );
        $this->add_responsive_control(
            'pre_image_dots_height',
            [
                'label' => __('Height', 'pixerex'),
                'type' => Controls_Manager::SLIDER,
                'size_units' => ['px', '%'],
                'range' => [
                    'px' => [
                        'min' => 0,
                        'max' => 1000,
                        'step' => 5,
                    ],
                    '%' => [
                        'min' => 0,
                        'max' => 100,
                    ],
                ],
                'devices' => ['desktop', 'tablet', 'mobile'],
                'default' => [
                    'unit' => 'px',
                    'size' => 100,
                ],
                'selectors' => [
                    '{{WRAPPER}} .pre-testimonial-gallary-pagination' => 'height: {{SIZE}}{{UNIT}} !important;',
                ],
            ]
        );

        $this->add_responsive_control(
            'pre_image_dots_image_size',
            [
                'label' => __('Image Size', 'pixerex'),
                'type' => Controls_Manager::SLIDER,
                'size_units' => ['px', '%'],
                'range' => [
                    'px' => [
                        'min' => 0,
                        'max' => 500,
                        'step' => 5,
                    ],
                    '%' => [
                        'min' => 0,
                        'max' => 100,
                    ],
                ],
                'devices' => ['desktop', 'tablet', 'mobile'],
                'default' => [
                    'unit' => 'px',
                    'size' => 100,
                ],
                'selectors' => [
                    '{{WRAPPER}} .pre-testimonial-gallary-pagination .pre-pagination-thumb .pre-thumbnail' => 'width: {{SIZE}}{{UNIT}} !important; height: {{SIZE}}{{UNIT}} !important;',
                ],
            ]
        );
        $this->add_control(
            'pre_image_dots_image_border_radius',
            [
                'label' => __('Border Radius', 'pixerex'),
                'type' => Controls_Manager::SLIDER,
                'size_units' => ['px', '%'],
                'range' => [
                    'px' => [
                        'min' => 0,
                        'max' => 1000,
                        'step' => 5,
                    ],
                    '%' => [
                        'min' => 0,
                        'max' => 100,
                    ],
                ],
                'selectors' => [
                    '{{WRAPPER}} .pre-testimonial-gallary-pagination .pre-pagination-thumb .pre-thumbnail' => 'border-radius: {{SIZE}}{{UNIT}};',
                ],
            ]
        );

        $this->end_controls_section();

    }

    protected function _render_user_meta( $item ) {
        $settings = $this->get_settings_for_display();
        ob_start();
?>
        <div class="pre-testimonial-meta">
            <p class="pre-testimonial-user"><?php echo esc_attr($item['pre_testimonial_name']); ?></p>
            <p class="pre-testimonial-user-company"><?php echo esc_attr($item['pre_testimonial_company_title']); ?></p>
        </div>
    <?php
        echo ob_get_clean();
    }

    protected function _render_user_avatar($item)
    {
        if ($item['pre_testimonial_enable_avatar'] != 'yes') return;
        $settings = $this->get_settings();
        ob_start();
    ?>
        <div class="pre-testimonial-image">
            <figure>
                <img src="<?php echo $item['pre_testimonial_image']['url']; ?>" alt="<?php echo esc_attr(get_post_meta($item['pre_testimonial_image']['id'], '_wp_attachment_image_alt', true)); ?>">
            </figure>
        </div>
    <?php
        echo ob_get_clean();
    }

    protected function _render_user_ratings($item)
    {
        if (empty($item['pre_testimonial_enable_rating'])) return;
        ob_start();
    ?>
        <ul class="testimonial-star-rating">
            <li><i class="fa fa-star" aria-hidden="true"></i></li>
            <li><i class="fa fa-star" aria-hidden="true"></i></li>
            <li><i class="fa fa-star" aria-hidden="true"></i></li>
            <li><i class="fa fa-star" aria-hidden="true"></i></li>
            <li><i class="fa fa-star" aria-hidden="true"></i></li>
        </ul>
    <?php
        echo ob_get_clean();
    }

    protected function _render_user_description($item)
    {
        echo '<div class="pre-testimonial-text">' . wpautop($item["pre_testimonial_description"]) . '</div>';
    }

    protected function render()
    {

        $settings = $this->get_settings_for_display();
        $testimonial_classes = $this->get_settings('pre_testimonial_image_rounded');
        $navigation_type = $this->get_settings('pre_testimonial_slider_navigation');
        $description_height = $this->get_settings('pre_testimonial_item_description_height');
        $img_position_class = $this->get_settings('pre_testimonial_img_position');
        $this->add_render_attribute('testimonial-slider-wrap', 'class', 'swiper-container-wrap');
        if($settings['navigation_arrow_position']=='outside-of-the-box'){
		    $this->add_render_attribute('testimonial-slider-wrap', 'class', 'pre-arrow-box');
        }

        if ($settings['dots_position']) {
            $this->add_render_attribute('testimonial-slider-wrap', 'class', 'swiper-container-wrap-dots-' . $settings['dots_position']);
        }

        $this->add_render_attribute('testimonial-slider-wrap', [
            'class' => ['pre-testimonial-slider'],
            'id'    => 'pre-testimonial-' . esc_attr($this->get_id()),
        ]);

        $this->add_render_attribute('testimonial-slider', [
            'class' => [
                'swiper-container',
                'pre-testimonial-slider-main',
                'swiper-container-' . esc_attr($this->get_id())
            ],
            'data-pagination'   => '.swiper-pagination-' . esc_attr($this->get_id()),
            'data-arrow-next'   => '.swiper-button-next-' . esc_attr($this->get_id()),
            'data-arrow-prev'   => '.swiper-button-prev-' . esc_attr($this->get_id())
        ]);

        if (!empty($settings['items']['size'])) {
            $this->add_render_attribute('testimonial-slider', 'data-items', $settings['items']['size']);
        }

        if (!empty($settings['items_tablet']['size'])) {
            $this->add_render_attribute('testimonial-slider', 'data-items-tablet', $settings['items_tablet']['size']);
        }

        if (!empty($settings['items_mobile']['size'])) {
            $this->add_render_attribute('testimonial-slider', 'data-items-mobile', $settings['items_mobile']['size']);
        }

        if (!empty($settings['margin']['size'])) {
            $this->add_render_attribute('testimonial-slider', 'data-margin', $settings['margin']['size']);
        }

        if (!empty($settings['margin_tablet']['size'])) {
            $this->add_render_attribute('testimonial-slider', 'data-margin-tablet', $settings['margin_tablet']['size']);
        }

        if (!empty($settings['margin_mobile']['size'])) {
            $this->add_render_attribute('testimonial-slider', 'data-margin-mobile', $settings['margin_mobile']['size']);
        }

        if ($settings['carousel_effect']) {
            $this->add_render_attribute('testimonial-slider', 'data-effect', $settings['carousel_effect']);
        }

        if (!empty($settings['slider_speed']['size'])) {
            $this->add_render_attribute('testimonial-slider', 'data-speed', $settings['slider_speed']['size']);
        }

        if ($settings['infinite_loop'] == 'yes') {
            $this->add_render_attribute('testimonial-slider', 'data-loop', 1);
        }

        if ($settings['grab_cursor'] == 'yes') {
            $this->add_render_attribute('testimonial-slider', 'data-grab-cursor', 1);
        }

        if ($settings['arrows'] == 'yes') {
            $this->add_render_attribute('testimonial-slider', 'data-arrows', 1);
        }

        if ($settings['dots'] == 'yes') {
            $this->add_render_attribute('testimonial-slider', 'data-dots', 1);
        }

        if ($settings['autoplay'] == 'yes' && !empty($settings['autoplay_speed']['size'])) {
            $this->add_render_attribute('testimonial-slider', 'data-autoplay_speed', $settings['autoplay_speed']['size']);
        }

        if ($settings['pause_on_hover'] == 'yes') {
            $this->add_render_attribute('testimonial-slider', 'data-pause-on-hover', 'true');
        }
    ?>

        <div <?php echo $this->get_render_attribute_string('testimonial-slider-wrap'); ?>>
            <?php
            if($settings['navigation_arrow_position']=='outside-of-the-box'){
	            $this->render_arrows();
            }
            ?>
            <?php if($settings['image_dots_position'] == 'above') { ?>
                <?php $this->render_image_dots(); ?>
            <?php } ?>
            <div <?php echo $this->get_render_attribute_string('testimonial-slider'); ?>>

                <div class="swiper-wrapper">
                    <?php
                    $i = 0;
                    foreach ($settings['pre_testimonial_slider_item'] as $item) :
                        $this->add_render_attribute('testimonial-content-wrapper' . $i, [
                            'class' => ['pre-testimonial-content', $item['pre_testimonial_rating_number']],
                            'style' => ($item['pre_testimonial_enable_avatar'] == '' ? 'width: 100%;' : '') . (!empty($description_height) ? ' height: ' . $description_height['size'] . $description_height['unit'] . ';'  : ''),
                        ]);

                        $this->add_render_attribute('testimonial-slide' . $i, [
                            'class' => ['pre-testimonial-item', 'clearfix', 'swiper-slide', $testimonial_classes]
                        ]);
                    ?>
                        <div <?php echo $this->get_render_attribute_string('testimonial-slide' . $i); ?>>
                            <div class="pre-testimonial-item-inner clearfix">
                                <div <?php echo $this->get_render_attribute_string('testimonial-content-wrapper' . $i); ?>>
                                    <?php if($settings['pre_testimonial_layout'] == 'top') { ?>
                                        <?php $this->_render_user_avatar($item); ?>
                                    <?php } ?>
                                    <?php if($settings['pre_testimonial_layout'] == 'above') { ?>
                                        <div class="testimonial-inline-style <?php echo $img_position_class; ?>">
                                            <?php
                                            $this->_render_user_avatar($item);
                                            $this->_render_user_meta($item);
                                            ?>
                                        </div>
                                    <?php } ?>
                                    <?php $this->_render_user_ratings($item); ?>
                                    <?php $this->_render_user_description($item); ?>
                                    <?php if($settings['pre_testimonial_layout'] == 'below') { ?>
                                        <div class="testimonial-inline-style <?php echo $img_position_class; ?>">
                                            <?php
                                            $this->_render_user_avatar($item);
                                            $this->_render_user_meta($item);
                                            ?>
                                        </div>
                                    <?php } ?>
                                    <?php if($settings['pre_testimonial_layout'] == 'top') { ?>
                                        <?php $this->_render_user_meta($item); ?>
                                    <?php } ?>  
                                </div>
                            </div>
                        </div>

                    <?php $i++;
                    endforeach; ?>
                </div>
                <?php
                $this->render_dots();
                if($settings['navigation_arrow_position']!='outside-of-the-box'){
	                $this->render_arrows();
                }
                ?>
            </div>
            <?php if($settings['image_dots_position'] == 'below') { ?>
                <?php $this->render_image_dots(); ?>
            <?php } ?>
        </div>

        <?php

    }

    /**
     * Render logo carousel dots output on the frontend.
     */
    protected function render_dots()
    {
        $settings = $this->get_settings_for_display();

        if ($settings['dots'] == 'yes' && $settings['image_dots'] != 'yes') { ?>
            <!-- Add Pagination -->
            <div class="swiper-pagination swiper-pagination-<?php echo esc_attr($this->get_id()); ?>"></div>
        <?php }
    }

    /**
     * Render logo carousel arrows output on the frontend.
     */
    protected function render_arrows()
    {
        $settings = $this->get_settings_for_display();

        if ($settings['arrows'] == 'yes') { ?>
            <?php
            if ($settings['arrow']) {
                $pa_next_arrow = $settings['arrow'];
                $pa_prev_arrow = str_replace("right", "left", $settings['arrow']);
            } else {
                $pa_next_arrow = 'fa fa-angle-right';
                $pa_prev_arrow = 'fa fa-angle-left';
            }
            ?>
            <!-- Add Arrows -->
            <div class="swiper-button-next swiper-button-next-<?php echo esc_attr($this->get_id()); ?>">
                <i class="<?php echo esc_attr($pa_next_arrow); ?>"></i>
            </div>
            <div class="swiper-button-prev swiper-button-prev-<?php echo esc_attr($this->get_id()); ?>">
                <i class="<?php echo esc_attr($pa_prev_arrow); ?>"></i>
            </div>
        <?php }
    }

    protected function render_image_dots()
    {
        $settings = $this->get_settings_for_display();

        $visibility = '';
        if ($settings['image_dots_visibility'] !== 'yes') {
            $visibility .= ' pre_gallery_pagination_hide_on_desktop';
        }
        if ($settings['image_dots_visibility_mobile'] !== 'yes') {
            $visibility .= ' pre_gallery_pagination_hide_on_mobile';
        }
        if ($settings['image_dots_visibility_tablet'] !== 'yes') {
            $visibility .= ' pre_gallery_pagination_hide_on_tablet';
        }

        $this->add_render_attribute('pre_gallery_pagination_wrapper', [
            'class' => ['swiper-container pre-testimonial-gallary-pagination', $visibility]
        ]);


        if ($settings['image_dots'] === 'yes') :
        ?>
            <div <?php echo $this->get_render_attribute_string('pre_gallery_pagination_wrapper'); ?>>
                <div class="swiper-wrapper">
                    <?php foreach ($settings['pre_testimonial_slider_item'] as $item) : ?>
                        <div class="swiper-slide">
                            <div class="swiper-slide-container">
                                <div class="pre-pagination-thumb">
                                    <img class="pre-thumbnail" src="<?php echo $item['pre_testimonial_image']['url']; ?>" alt="<?php echo esc_attr(get_post_meta($item['pre_testimonial_image']['id'], '_wp_attachment_image_alt', true)); ?>">
                                </div>
                            </div>
                        </div>
                    <?php endforeach; ?>
                </div>
            </div>
<?php
        endif;
    }

    protected function content_template()
    {
    }
}

Plugin::instance()->widgets_manager->register( new PR_TestimonialSlider_Widget() );